<?php
// ============================================================
//  public/contact.php — Contact form (display + handler)
// ============================================================
require_once dirname(__DIR__) . '/app/config.php';
require_once ROOT_PATH . '/app/db.php';
require_once ROOT_PATH . '/app/functions.php';

$errors  = [];
$success = false;
$old     = []; // repopulate form on error

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    if (!csrf_verify()) {
        $errors[] = 'Invalid form submission. Please refresh and try again.';
    } else {
        $old = [
            'name'          => trim($_POST['name']          ?? ''),
            'email'         => trim($_POST['email']         ?? ''),
            'audience_type' => trim($_POST['audience_type'] ?? ''),
            'message'       => trim($_POST['message']       ?? ''),
        ];

        // Validation
        if (strlen($old['name']) < 2)
            $errors[] = 'Please enter your name.';
        if (!filter_var($old['email'], FILTER_VALIDATE_EMAIL))
            $errors[] = 'Please enter a valid email address.';
        if (strlen($old['message']) < 20)
            $errors[] = 'Message must be at least 20 characters.';

        // Honeypot — bots fill hidden fields, humans don't
        if (!empty($_POST['website'])) {
            // Silently succeed to fool bots
            $success = true;
        }

        if (empty($errors) && !$success) {
            db_insert(
                "INSERT INTO contact_messages (name, email, audience_type, message) VALUES (?,?,?,?)",
                [$old['name'], $old['email'], $old['audience_type'], $old['message']]
            );

            // Notify admin by email
            $subject = '[GetPhDReady] New contact message from ' . $old['name'];
            $body    = "Name: {$old['name']}\n"
                     . "Email: {$old['email']}\n"
                     . "Audience: {$old['audience_type']}\n\n"
                     . "Message:\n{$old['message']}\n\n"
                     . "Reply at: " . SITE_URL . "/admin/contact_messages/index.php";
            @mail(ADMIN_EMAIL, $subject, $body,
                  "From: " . SITE_NAME . " <" . SITE_EMAIL . ">\r\nContent-Type: text/plain; charset=UTF-8\r\n");

            $success = true;
            $old     = []; // clear form
        }
    }
}

$pageTitle = 'Contact';
$pageDesc  = 'Get in touch with the GetPhDReady team.';
$activeNav = 'contact';
include TEMPLATE_PATH . '/header.php';
?>

<section id="contact" style="padding-top:8rem;">
  <div class="reveal">
    <span class="section-label">Contact</span>
    <h2 class="section-heading">Get in <em>Touch</em></h2>
    <p class="section-sub">Have a question, want to collaborate, or just want to say hello? Fill in the form and we'll get back to you.</p>
  </div>

  <div class="contact-layout reveal">
    <!-- Contact info -->
    <div class="contact-info">
      <div class="contact-item">
        <h4>General Inquiries</h4>
        <p><?= h(SITE_EMAIL) ?></p>
      </div>
      <div class="contact-item">
        <h4>Collaborations &amp; Partnerships</h4>
        <p>partners@getphdready.com</p>
      </div>
      <div class="contact-item">
        <h4>Content Suggestions</h4>
        <p>Have a topic you'd like covered? We welcome suggestions from our community.</p>
      </div>
      <div class="contact-item">
        <h4>Follow Along</h4>
        <p>
          <a href="https://twitter.com/getphdready" target="_blank" rel="noopener">Twitter / X</a> ·
          <a href="https://linkedin.com/company/getphdready" target="_blank" rel="noopener">LinkedIn</a> ·
          <a href="https://youtube.com/@getphdready" target="_blank" rel="noopener">YouTube</a>
        </p>
      </div>
    </div>

    <!-- Contact form -->
    <div class="contact-form">
      <?php if ($success): ?>
        <div class="alert alert--success" style="margin-bottom:2rem;">
          ✓ Message sent! We'll get back to you within a few days.
        </div>
      <?php endif; ?>

      <?php if ($errors): ?>
        <div class="alert alert--error" style="margin-bottom:1.5rem;">
          <?php foreach ($errors as $e): ?>
            <p><?= h($e) ?></p>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>

      <?php if (!$success): ?>
      <form method="POST" action="<?= SITE_URL ?>/contact.php">
        <?= csrf_field() ?>
        <!-- Honeypot (hidden from real users with CSS) -->
        <input type="text" name="website" style="display:none;" tabindex="-1" autocomplete="off">

        <div class="form-row">
          <div class="form-field">
            <label>First Name</label>
            <input type="text" name="name" value="<?= h($old['name'] ?? '') ?>"
                   placeholder="Jane" required maxlength="100">
          </div>
          <div class="form-field">
            <label>Email</label>
            <input type="email" name="email" value="<?= h($old['email'] ?? '') ?>"
                   placeholder="jane@university.edu" required maxlength="150">
          </div>
        </div>
        <div class="form-field">
          <label>I am a…</label>
          <select name="audience_type">
            <option value="">Select…</option>
            <?php foreach (['Future PhD Applicant','Current PhD Student','Postdoc / Early Career Researcher','Faculty / Administrator','Other'] as $opt): ?>
            <option value="<?= h($opt) ?>" <?= ($old['audience_type'] ?? '') === $opt ? 'selected' : '' ?>>
              <?= h($opt) ?>
            </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="form-field">
          <label>Message</label>
          <textarea name="message" rows="6" placeholder="What's on your mind?"
                    required minlength="20" maxlength="3000"><?= h($old['message'] ?? '') ?></textarea>
        </div>
        <button type="submit" class="btn-primary">Send Message</button>
      </form>
      <?php endif; ?>
    </div>
  </div>
</section>

<?php include TEMPLATE_PATH . '/footer.php'; ?>
