<?php
// ============================================================
//  admin/videos/create.php + edit.php (shared)
// ============================================================
require_once dirname(__DIR__, 2) . '/app/config.php';
require_once ROOT_PATH . '/app/db.php';
require_once ROOT_PATH . '/app/functions.php';
require_editor();

$editMode = str_ends_with($_SERVER['PHP_SELF'], 'edit.php');
$videoId  = $editMode ? (int)($_GET['id'] ?? 0) : 0;

$video = null;
if ($editMode) {
    $video = db_one("SELECT * FROM videos WHERE id = ?", [$videoId]);
    if (!$video) redirect_with_message(SITE_URL . '/admin/videos/index.php', 'Video not found.', 'error');
}

$errors = [];
$data   = [
    'title'            => $video['title']            ?? '',
    'slug'             => $video['slug']             ?? '',
    'description'      => $video['description']      ?? '',
    'youtube_url'      => $video['youtube_id'] ? 'https://www.youtube.com/watch?v='.$video['youtube_id'] : '',
    'youtube_id'       => $video['youtube_id']       ?? '',
    'host'             => $video['host']             ?? '',
    'category_id'      => $video['category_id']      ?? '',
    'duration_seconds' => $video['duration_seconds'] ?? 0,
    'video_type'       => $video['video_type']       ?? 'tutorial',
    'audience'         => $video['audience']         ?? 'both',
    'status'           => $video['status']           ?? 'draft',
    'featured'         => $video['featured']         ?? 0,
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!csrf_verify()) $errors[] = 'Invalid token.';

    $youtubeUrl = trim($_POST['youtube_url'] ?? '');
    $youtubeId  = youtube_id_from_url($youtubeUrl) ?? trim($youtubeUrl); // allow bare ID too

    // Convert "mm:ss" or "h:mm:ss" duration input to seconds
    $durationInput = trim($_POST['duration'] ?? '');
    $durationSecs  = 0;
    if ($durationInput) {
        $parts = array_reverse(explode(':', $durationInput));
        $durationSecs = (int)($parts[0] ?? 0)
                      + (int)($parts[1] ?? 0) * 60
                      + (int)($parts[2] ?? 0) * 3600;
    }

    $data = [
        'title'            => trim($_POST['title']       ?? ''),
        'slug'             => trim($_POST['slug']        ?? ''),
        'description'      => trim($_POST['description'] ?? ''),
        'youtube_url'      => $youtubeUrl,
        'youtube_id'       => $youtubeId,
        'host'             => trim($_POST['host']        ?? ''),
        'category_id'      => (int)($_POST['category_id'] ?? 0) ?: null,
        'duration_seconds' => $durationSecs,
        'video_type'       => in_array($_POST['video_type']??'', ['tutorial','interview','panel','workshop','guide','q_and_a']) ? $_POST['video_type'] : 'tutorial',
        'audience'         => in_array($_POST['audience']??'', ['applicants','current','both']) ? $_POST['audience'] : 'both',
        'status'           => in_array($_POST['status']??'', ['draft','published'])             ? $_POST['status']   : 'draft',
        'featured'         => isset($_POST['featured']) ? 1 : 0,
    ];

    if (!$data['slug']) $data['slug'] = slugify($data['title']);
    $data['slug'] = unique_slug('videos', slugify($data['slug']), $videoId);

    if (strlen($data['title'])       < 5)  $errors[] = 'Title is required.';
    if (!$data['youtube_id'])               $errors[] = 'A valid YouTube URL or video ID is required.';
    if (strlen($data['description']) < 20) $errors[] = 'Description must be at least 20 characters.';

    if (empty($errors)) {
        $publishedAt = $data['status'] === 'published'
            ? ($video['published_at'] ?? date('Y-m-d H:i:s'))
            : null;

        if ($editMode) {
            db_run("
                UPDATE videos SET title=?,slug=?,description=?,youtube_id=?,host=?,
                category_id=?,duration_seconds=?,video_type=?,audience=?,
                status=?,featured=?,published_at=?,updated_at=NOW() WHERE id=?
            ", [
                $data['title'],$data['slug'],$data['description'],$data['youtube_id'],
                $data['host'],$data['category_id'],$data['duration_seconds'],
                $data['video_type'],$data['audience'],$data['status'],
                $data['featured'],$publishedAt,$videoId
            ]);
            redirect_with_message(SITE_URL . '/admin/videos/index.php', 'Video updated.', 'success');
        } else {
            $newId = db_insert("
                INSERT INTO videos (title,slug,description,youtube_id,host,category_id,
                duration_seconds,video_type,audience,status,featured,published_at)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
            ", [
                $data['title'],$data['slug'],$data['description'],$data['youtube_id'],
                $data['host'],$data['category_id'],$data['duration_seconds'],
                $data['video_type'],$data['audience'],$data['status'],
                $data['featured'],$publishedAt
            ]);
            redirect_with_message(SITE_URL . '/admin/videos/edit.php?id='.$newId, 'Video created!', 'success');
        }
    }
}

$categories = db_all("SELECT id, name FROM categories WHERE type='video' ORDER BY sort_order");
$adminTitle = $editMode ? 'Edit Video' : 'New Video';
$ADMIN_TPL  = ROOT_PATH . '/admin/templates';
include $ADMIN_TPL . '/admin_header.php';
?>

<form method="POST" class="admin-form">
  <?= csrf_field() ?>

  <?php if ($errors): ?>
  <div class="alert alert--error">
    <?php foreach ($errors as $e): ?><p><?= h($e) ?></p><?php endforeach; ?>
  </div>
  <?php endif; ?>

  <div class="form-layout-two-col">
    <div class="form-main">
      <div class="form-field">
        <label for="title">Title <span class="required">*</span></label>
        <input type="text" id="title" name="title" value="<?= h($data['title']) ?>"
               required maxlength="255" oninput="autoSlug(this.value)">
      </div>
      <div class="form-field">
        <label for="slug">Slug</label>
        <input type="text" id="slug" name="slug" value="<?= h($data['slug']) ?>" maxlength="280">
      </div>
      <div class="form-field">
        <label for="youtube_url">YouTube URL <span class="required">*</span></label>
        <input type="text" id="youtube_url" name="youtube_url"
               value="<?= h($data['youtube_url']) ?>"
               placeholder="https://www.youtube.com/watch?v=XXXXXXXXXXX"
               oninput="previewThumb(this.value)">
        <small class="field-hint">Paste the full YouTube URL — the ID is extracted automatically.</small>
      </div>

      <!-- Live YouTube thumbnail preview -->
      <div id="thumbPreview" style="<?= $data['youtube_id'] ? '' : 'display:none;' ?>margin-bottom:1.5rem;">
        <img id="thumbImg"
             src="<?= $data['youtube_id'] ? h(youtube_thumb_url($data['youtube_id'])) : '' ?>"
             alt="YouTube thumbnail preview"
             style="max-width:320px;border-radius:3px;border:1px solid var(--dark-border);">
      </div>

      <div class="form-field">
        <label for="description">Description <span class="required">*</span></label>
        <textarea id="description" name="description" rows="8"
                  required minlength="20"><?= h($data['description']) ?></textarea>
      </div>
      <div class="form-field">
        <label for="host">Host / Speaker</label>
        <input type="text" id="host" name="host" value="<?= h($data['host']) ?>"
               placeholder="e.g. Dr. Sarah Chen" maxlength="150">
      </div>
    </div>

    <div class="form-sidebar">
      <div class="admin-card meta-card">
        <h3>Publish</h3>
        <div class="form-field">
          <label>Status</label>
          <select name="status">
            <option value="draft"     <?= $data['status']==='draft'     ?'selected':'' ?>>Draft</option>
            <option value="published" <?= $data['status']==='published' ?'selected':'' ?>>Published</option>
          </select>
        </div>
        <div class="form-check">
          <input type="checkbox" id="featured" name="featured" value="1" <?= $data['featured']?'checked':'' ?>>
          <label for="featured">Feature on homepage</label>
        </div>
        <button type="submit" class="admin-btn admin-btn--primary" style="width:100%;margin-top:1rem;">
          <?= $editMode ? 'Update Video' : 'Create Video' ?>
        </button>
        <a href="<?= SITE_URL ?>/admin/videos/index.php"
           class="admin-btn admin-btn--ghost" style="width:100%;text-align:center;margin-top:.5rem;">Cancel</a>
      </div>

      <div class="admin-card meta-card">
        <h3>Details</h3>
        <div class="form-field">
          <label for="video_type">Video Type</label>
          <select id="video_type" name="video_type">
            <?php foreach (['tutorial'=>'Tutorial','interview'=>'Interview','panel'=>'Panel Discussion','workshop'=>'Workshop','guide'=>'Guide','q_and_a'=>'Q & A'] as $val => $lbl): ?>
            <option value="<?= $val ?>" <?= $data['video_type']===$val?'selected':'' ?>><?= $lbl ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="form-field">
          <label for="category_id">Category</label>
          <select id="category_id" name="category_id">
            <option value="">— None —</option>
            <?php foreach ($categories as $c): ?>
            <option value="<?= $c['id'] ?>" <?= $data['category_id']==$c['id']?'selected':'' ?>><?= h($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="form-field">
          <label for="audience">Audience</label>
          <select id="audience" name="audience">
            <option value="both"       <?= $data['audience']==='both'       ?'selected':'' ?>>Everyone</option>
            <option value="applicants" <?= $data['audience']==='applicants' ?'selected':'' ?>>Future Applicants</option>
            <option value="current"    <?= $data['audience']==='current'    ?'selected':'' ?>>Current Students</option>
          </select>
        </div>
        <div class="form-field">
          <label for="duration">Duration (h:mm:ss or mm:ss)</label>
          <input type="text" id="duration" name="duration"
                 value="<?php
                    $s = $data['duration_seconds'];
                    $h = intdiv($s,3600); $m = intdiv($s%3600,60); $sec = $s%60;
                    echo $h ? sprintf('%d:%02d:%02d',$h,$m,$sec) : sprintf('%d:%02d',$m,$sec);
                 ?>"
                 placeholder="e.g. 52:14">
        </div>
      </div>
    </div>
  </div>
</form>

<script>
let slugManual = <?= json_encode($editMode) ?>;
function autoSlug(t) {
  if (slugManual) return;
  document.getElementById('slug').value = t.toLowerCase()
    .replace(/[^a-z0-9\s-]/g,'').trim().replace(/\s+/g,'-').replace(/-+/g,'-');
}
document.getElementById('slug').addEventListener('input', () => slugManual = true);

function previewThumb(url) {
  const match = url.match(/(?:v=|youtu\.be\/|embed\/)([a-zA-Z0-9_-]{11})/);
  const wrap = document.getElementById('thumbPreview');
  const img  = document.getElementById('thumbImg');
  if (match) {
    img.src = 'https://img.youtube.com/vi/' + match[1] + '/hqdefault.jpg';
    wrap.style.display = '';
  } else {
    wrap.style.display = 'none';
  }
}
</script>

<?php include $ADMIN_TPL . '/admin_footer.php'; ?>
