<?php
// ============================================================
//  admin/articles/create.php  — create new article
//  admin/articles/edit.php    — edit existing article
//  Both use this same form logic; edit.php just sets $editMode.
// ============================================================
require_once dirname(__DIR__, 2) . '/app/config.php';
require_once ROOT_PATH . '/app/db.php';
require_once ROOT_PATH . '/app/functions.php';
require_editor();

// Detect edit vs create
$editMode  = str_ends_with($_SERVER['PHP_SELF'], 'edit.php');
$articleId = $editMode ? (int)($_GET['id'] ?? 0) : 0;

// Load existing article for edit
$article = null;
if ($editMode) {
    $article = db_one("SELECT * FROM articles WHERE id = ?", [$articleId]);
    if (!$article) {
        redirect_with_message(SITE_URL . '/admin/articles/index.php', 'Article not found.', 'error');
    }
}

$errors = [];
$data   = [
    'title'       => $article['title']       ?? '',
    'slug'        => $article['slug']        ?? '',
    'excerpt'     => $article['excerpt']     ?? '',
    'body'        => $article['body']        ?? '',
    'category_id' => $article['category_id'] ?? '',
    'audience'    => $article['audience']    ?? 'both',
    'status'      => $article['status']      ?? 'draft',
    'featured'    => $article['featured']    ?? 0,
    'read_time'   => $article['read_time']   ?? 5,
    'thumbnail'   => $article['thumbnail']   ?? '',
];

// ── Handle form submit ────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!csrf_verify()) $errors[] = 'Invalid form token.';

    $data = [
        'title'       => trim($_POST['title']       ?? ''),
        'slug'        => trim($_POST['slug']        ?? ''),
        'excerpt'     => trim($_POST['excerpt']     ?? ''),
        'body'        => $_POST['body']             ?? '',  // raw HTML from editor
        'category_id' => (int)($_POST['category_id'] ?? 0) ?: null,
        'audience'    => in_array($_POST['audience']??'', ['applicants','current','both']) ? $_POST['audience'] : 'both',
        'status'      => in_array($_POST['status']??'',   ['draft','published'])           ? $_POST['status']   : 'draft',
        'featured'    => isset($_POST['featured']) ? 1 : 0,
        'read_time'   => read_time($_POST['body'] ?? ''),
        'thumbnail'   => $article['thumbnail'] ?? '',
    ];

    // Auto-generate slug from title if blank
    if (!$data['slug']) $data['slug'] = slugify($data['title']);
    $data['slug'] = unique_slug('articles', slugify($data['slug']), $articleId);

    // Validation
    if (strlen($data['title'])   < 5)  $errors[] = 'Title must be at least 5 characters.';
    if (strlen($data['excerpt']) < 20) $errors[] = 'Excerpt must be at least 20 characters.';
    if (strlen($data['body'])    < 50) $errors[] = 'Body must be at least 50 characters.';

    // Handle thumbnail upload
    if (!empty($_FILES['thumbnail']['name'])) {
        try {
            $data['thumbnail'] = upload_image($_FILES['thumbnail'], 'articles');
        } catch (RuntimeException $e) {
            $errors[] = 'Thumbnail: ' . $e->getMessage();
        }
    }

    if (empty($errors)) {
        $publishedAt = ($data['status'] === 'published')
            ? ($article['published_at'] ?? date('Y-m-d H:i:s'))
            : null;

        if ($editMode) {
            db_run("
                UPDATE articles
                SET title=?, slug=?, excerpt=?, body=?, thumbnail=?,
                    category_id=?, audience=?, status=?, featured=?,
                    read_time=?, published_at=?, updated_at=NOW()
                WHERE id=?
            ", [
                $data['title'], $data['slug'], $data['excerpt'], $data['body'],
                $data['thumbnail'], $data['category_id'], $data['audience'],
                $data['status'], $data['featured'], $data['read_time'],
                $publishedAt, $articleId
            ]);
            redirect_with_message(SITE_URL . '/admin/articles/index.php', 'Article updated successfully.', 'success');
        } else {
            $newId = db_insert("
                INSERT INTO articles
                (title, slug, excerpt, body, thumbnail, author_id, category_id,
                 audience, status, featured, read_time, published_at)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
            ", [
                $data['title'], $data['slug'], $data['excerpt'], $data['body'],
                $data['thumbnail'], $_SESSION['user_id'], $data['category_id'],
                $data['audience'], $data['status'], $data['featured'],
                $data['read_time'], $publishedAt
            ]);
            redirect_with_message(SITE_URL . '/admin/articles/edit.php?id=' . $newId, 'Article created! You can continue editing.', 'success');
        }
    }
}

$categories = db_all("SELECT id, name FROM categories WHERE type='article' ORDER BY sort_order");
$adminTitle = $editMode ? 'Edit Article' : 'New Article';
$ADMIN_TPL  = ROOT_PATH . '/admin/templates';
include $ADMIN_TPL . '/admin_header.php';
?>

<form method="POST" enctype="multipart/form-data" class="admin-form">
  <?= csrf_field() ?>

  <?php if ($errors): ?>
  <div class="alert alert--error">
    <?php foreach ($errors as $e): ?><p><?= h($e) ?></p><?php endforeach; ?>
  </div>
  <?php endif; ?>

  <div class="form-layout-two-col">

    <!-- ── LEFT: main fields ── -->
    <div class="form-main">
      <div class="form-field">
        <label for="title">Title <span class="required">*</span></label>
        <input type="text" id="title" name="title"
               value="<?= h($data['title']) ?>"
               placeholder="Article title" required maxlength="255"
               oninput="autoSlug(this.value)">
      </div>

      <div class="form-field">
        <label for="slug">Slug <span class="required">*</span></label>
        <div class="slug-row">
          <span class="slug-prefix"><?= SITE_URL ?>/article.php?slug=</span>
          <input type="text" id="slug" name="slug"
                 value="<?= h($data['slug']) ?>"
                 maxlength="280" placeholder="auto-generated-from-title">
        </div>
      </div>

      <div class="form-field">
        <label for="excerpt">Excerpt <span class="required">*</span></label>
        <textarea id="excerpt" name="excerpt" rows="3"
                  placeholder="Short summary shown in article listings…"
                  required minlength="20" maxlength="500"><?= h($data['excerpt']) ?></textarea>
        <small class="field-hint">Shown on article cards and in meta description.</small>
      </div>

      <div class="form-field">
        <label for="body">Body <span class="required">*</span></label>
        <!-- Basic textarea — swap for a rich-text editor (TinyMCE / Quill)
             by pointing the editor's output to name="body" -->
        <textarea id="body" name="body" rows="24"
                  class="body-editor"
                  placeholder="Full article content (HTML supported)…"><?= h($data['body']) ?></textarea>
        <small class="field-hint">HTML is allowed. Read time auto-calculated on save.</small>
      </div>
    </div>

    <!-- ── RIGHT: meta fields ── -->
    <div class="form-sidebar">

      <!-- Publish box -->
      <div class="admin-card meta-card">
        <h3>Publish</h3>
        <div class="form-field">
          <label for="status">Status</label>
          <select id="status" name="status">
            <option value="draft"     <?= $data['status']==='draft'     ?'selected':'' ?>>Draft</option>
            <option value="published" <?= $data['status']==='published' ?'selected':'' ?>>Published</option>
          </select>
        </div>
        <div class="form-check">
          <input type="checkbox" id="featured" name="featured" value="1"
                 <?= $data['featured'] ? 'checked' : '' ?>>
          <label for="featured">Feature on homepage</label>
        </div>
        <div class="meta-actions">
          <button type="submit" class="admin-btn admin-btn--primary" style="width:100%;">
            <?= $editMode ? 'Update Article' : 'Create Article' ?>
          </button>
          <a href="<?= SITE_URL ?>/admin/articles/index.php" class="admin-btn admin-btn--ghost" style="width:100%;text-align:center;">Cancel</a>
        </div>
      </div>

      <!-- Category + audience -->
      <div class="admin-card meta-card">
        <h3>Categorise</h3>
        <div class="form-field">
          <label for="category_id">Category</label>
          <select id="category_id" name="category_id">
            <option value="">— None —</option>
            <?php foreach ($categories as $cat): ?>
            <option value="<?= $cat['id'] ?>" <?= $data['category_id']==$cat['id']?'selected':'' ?>>
              <?= h($cat['name']) ?>
            </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="form-field">
          <label for="audience">Audience</label>
          <select id="audience" name="audience">
            <option value="both"       <?= $data['audience']==='both'       ?'selected':'' ?>>Everyone</option>
            <option value="applicants" <?= $data['audience']==='applicants' ?'selected':'' ?>>Future Applicants</option>
            <option value="current"    <?= $data['audience']==='current'    ?'selected':'' ?>>Current PhD Students</option>
          </select>
        </div>
      </div>

      <!-- Thumbnail -->
      <div class="admin-card meta-card">
        <h3>Thumbnail</h3>
        <?php if ($data['thumbnail']): ?>
        <img src="<?= h(UPLOAD_URL . '/' . $data['thumbnail']) ?>"
             alt="Current thumbnail"
             class="thumb-preview">
        <?php endif; ?>
        <div class="form-field">
          <label for="thumbnail">Upload image</label>
          <input type="file" id="thumbnail" name="thumbnail"
                 accept="image/jpeg,image/png,image/webp">
          <small class="field-hint">JPG, PNG or WebP · max 5 MB</small>
        </div>
      </div>

    </div>
  </div>
</form>

<script>
// Auto-fill slug from title (only when slug is empty or was auto-generated)
let slugManual = <?= json_encode($editMode) ?>;
function autoSlug(title) {
  if (slugManual) return;
  document.getElementById('slug').value = title
    .toLowerCase()
    .replace(/[^a-z0-9\s-]/g, '')
    .trim()
    .replace(/[\s]+/g, '-')
    .replace(/-+/g, '-');
}
document.getElementById('slug').addEventListener('input', () => slugManual = true);
</script>

<?php include $ADMIN_TPL . '/admin_footer.php'; ?>
